<?php

class Cusmin_Post_Handler
{
    protected $action;
    protected $data;
    protected $isUserAdmin = false;

    public function __construct($post, $isAdmin = false)
    {
        $this->data = $post;
        $this->action = (is_array($post) && isset($post['action']))?$post['action']:'';
        $this->isUserAdmin = $isAdmin;
    }

    public function handle()
    {
        //User methods
        switch ($this->action) {
            case Cusmin_Post_Action::NOTICES_LOAD:
                return $this->NoticesLoad();
        }

        if(!$this->isUserAdmin){
            echo 'You are not administrator.';
        }

        //Admin methods
        switch ($this->action) {
            case Cusmin_Post_Action::CONFIGURATION_CREATE:
                return $this->configurationCreate();
            case Cusmin_Post_Action::CONFIGURATION_DELETE:
                return $this->configurationDelete();
            case Cusmin_Post_Action::CONFIGURATION_UPDATE_NAME:
                return $this->configurationUpdateName();
            case Cusmin_Post_Action::CONFIGURATION_GET_ONE:
                return $this->configurationOne();
            case Cusmin_Post_Action::CONFIGURATION_GET_ALL:
                return $this->configurationAll();
            case Cusmin_Post_Action::CONFIGURATION_DEFAULT_GET_NAME:
                return $this->configurationDefaultGetName();
            case Cusmin_Post_Action::CONFIGURATION_DEFAULT_SET_NAME:
                return $this->configurationDefaultSetName();
            case Cusmin_Post_Action::TRIAL_DISABLE:
                return $this->trialDisable();
            case Cusmin_Post_Action::TRIAL_ENABLE:
                return $this->trialEnable();

            case Cusmin_Post_Action::CUSTOMIZATION_SAVE:
                return $this->customizationSave();
            case Cusmin_Post_Action::CUSTOMIZATION_CLEAR:
                return $this->customizationClear();
            case Cusmin_Post_Action::CUSTOMIZATION_REMOVE:
                return $this->customizationRemove();

            case Cusmin_Post_Action::ACCESS_BY_SAVE:
                return $this->AccessBySave();
            case Cusmin_Post_Action::ACCESS_BY_RESET:
                return $this->AccessByReset();
            case Cusmin_Post_Action::APPLY_TO_SAVE:
                return $this->applyToSave();
            case Cusmin_Post_Action::APPLY_TO_RESET:
                return $this->ApplyToReset();
            case Cusmin_Post_Action::APPLY_TO_GET_ONE:
                return $this->ApplyToGetOne();
            case Cusmin_Post_Action::APPLY_TO_SEARCH_USERS:
                return $this->ApplyToSearchUsers();
            case Cusmin_Post_Action::GET_COLUMNS:
                return $this->GetColumns();
            case Cusmin_Post_Action::GET_META_BOXES:
                return $this->GetMetaBoxes();
            case Cusmin_Post_Action::CONFIGURATIONS_REORDER:
                return $this->ConfigurationsReorder();
            case Cusmin_Post_Action::NOTICES_CREATE:
                return $this->NoticesCreate();
            case Cusmin_Post_Action::NOTICES_DELETE:
                return $this->NoticesDelete();
            case Cusmin_Post_Action::DASHBOARD_WIDGETS_ORDERING:
                return $this->DashboardWidgetsOrdering();
            case Cusmin_Post_Action::CUSTOMIZATIONS_APPLICATION_MODE:
                return $this->CustomizationsApplicationMode();
            case Cusmin_Post_Action::LOAD_AGCA_SETTINGS:
                return $this->loadAGCASettings();

            //API
            case Cusmin_Post_Action::API_TEST:
                return $this->apiTest();

            //Resets everything to defaults
            case Cusmin_Post_Action::RESET_ALL:
                return $this->resetAll();

            case Cusmin_Post_Action::LOGIN_CONFIGURATION:
                return $this->loginSelectedConfiguration();
        }
    }

    /**
     * Creates new configuration if it does not exist already
     * It creates also default empty customization
     */
    protected function configurationCreate()
    {
        $slug = $this->data['slug'];
        $name = $this->data['name'];

        try {
//            $this->mustBeSlugFormat($slug, '(Slug)');
//            $this->mustBeSlugFormat($name, '(Name)');

            CusminOptions::addConfiguration($slug, $name);
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Deletes existing configuration with all customizations in it
     */
    protected function configurationDelete()
    {
        $slug = $this->data['slug'];
        try {
            CusminOptions::removeConfiguration($slug);
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    /**
     * It updates only configuration name and
     * authentication rules
     */
    protected function configurationUpdateName()
    {
        $oldSlug = $this->data['oldSlug'];
        $newName = $this->data['newName'];
        try {
            CusminOptions::updateConfigurationName($oldSlug, $newName);
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function configurationOne()
    {
        $slug = $this->data['slug'];
        try {
            $c = CusminOptions::getOneConfiguration($slug);
            return $this->success(null, $c);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function configurationAll()
    {
        try {
            $c = CusminOptions::getAllConfigurations();
            return $this->success(null, $c);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function configurationDefaultGetName()
    {
        try {
            return $this->success(null, CusminOptions::getDefaultConfigurationSlug());
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function configurationDefaultSetName()
    {
        try {
            $slug = $this->data['slug'];
            $name = $this->data['name'];
            CusminOptions::setDefaultConfigurationName($slug, $name);
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function trialDisable()
    {
        try {
            CusminOptions::toggleTrial(true);
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function trialEnable()
    {
        try {
            CusminOptions::toggleTrial(false);
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function loginSelectedConfiguration()
    {
        try {
            $configurationSlug = $this->data['slug'];
            $configurationName = $this->data['name'];
            CusminOptions::setLoginSelectedConfiguration($configurationSlug, $configurationName);
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Creates new customization inside existing configuration
     */
    protected function customizationSave()
    {
        $configurationSlug = $this->data['configurationSlug'];
        $configurationName = $this->data['configurationName'];
        $data = $this->data['data'];
        try {
            CusminOptions::saveCustomization($configurationSlug, $configurationName, $data);
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }


    /**
     * Deletes customization data from existing configuration
     */
    protected function customizationClear()
    {
        $configurationSlug = $this->data['configurationSlug'];
        try {
            CusminOptions::clearCustomization($configurationSlug);
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function customizationRemove()
    {
        $configurationSlug = $this->data['configurationSlug'];
        try {
            CusminOptions::removeCustomization($configurationSlug);
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function AccessBySave()
    {
        $configurationSlug = $this->data['configurationSlug'];
        $data = $this->data['data'];
        try {
            CusminOptions::saveAccessBy($configurationSlug, $data);
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function AccessByReset()
    {
        $configurationSlug = $this->data['configurationSlug'];
        try {
            CusminOptions::resetAccessBy($configurationSlug);
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function ApplyToSave()
    {
        $configurationSlug = $this->data['configurationSlug'];
        $configurationName = $this->data['configurationName'];
        try {
            if(isset($this->data['data'])) {
                $data = $this->data['data'];
                CusminOptions::saveApplyTo($configurationSlug, $configurationName, $data);
                return $this->success(null, null);
            }else {
                CusminOptions::resetApplyTo($configurationSlug);
                return $this->success(null, null);
                //return $this->fail('Cannot remove the last rule, at least one rule should exist. Please add another to remove this one', '400');
            }
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function ApplyToReset()
    {
        $configurationSlug = $this->data['configurationSlug'];
        try {
            CusminOptions::resetApplyTo($configurationSlug);
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function ApplyToGetOne()
    {
        $configurationSlug = $this->data['configurationSlug'];
        try {
            $applyTo = CusminOptions::getOneApplyTo($configurationSlug);
            return $this->success(null, $applyTo);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function ApplyToSearchUsers()
    {
        $search = $this->data['search'];
        try {
            $users = CusminOptions::applyToSearchUsers($search);
            return $this->success(null, $users);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function getPostTypeUrl($postType){
            switch ($postType){
                case 'revision':case 'attachment':case 'nav_menu_item':case 'custom_css':case 'customize_changeset':
                //skip these post types
                return null;
                case 'plugins':
                    return 'plugins.php';
                case 'comments':
                    return 'edit-comments.php';
                case 'users':
                    return 'users.php';
            }
            return 'edit.php?post_type=' . $postType;
    }

    protected function GetColumns()
    {
        try {
            ini_set('display_errors', 0);
            set_time_limit(0);

            $forceReload = $this->data['forceReload'];

            if( $forceReload !== 'true' ){
                $columnsCached = get_transient('cusmin-columns');
                if($columnsCached) {
                    return $this->success(null, $columnsCached);
                }
            }

            $postTypes = $this->data['postTypes'];
            $data = array();
            foreach ((array) $postTypes as $type){
                $url = get_site_url() . '/wp-admin/' . $this->getPostTypeUrl($type). '&cusmin-action=cusmin-list-columns';

                // Create a stream
                $opts = [
                    "http" => [
                        "method" => "GET",
                        "header" =>
                            "Accept-language: ".$_SERVER['HTTP_ACCEPT_LANGUAGE']."\r\n" .
                            "Cookie: ".$_SERVER['HTTP_COOKIE']."\r\n"
                    ]
                ];

                // DOCS: https://www.php.net/manual/en/function.stream-context-create.php
                $context = stream_context_create($opts);

                // Open the file using the HTTP headers set above
                // DOCS: https://www.php.net/manual/en/function.file-get-contents.php

                try{
                    $content = file_get_contents($url, false, $context);

                    $m = substr($content, strpos($content, '<posts-columns-list>')+20);
                    $m = substr($m, 0, strpos($m, '</posts-columns-list>'));

                    if(!empty($m)) {
                        $data[$type] = json_decode($m);
                    }
                }catch (\Exception $e){
                    //$data[$type] = 'error: ' .$e->getMessage();
                }
            }
            $response = ['time' => current_time('mysql'), 'columns' => $data];
            set_transient('cusmin-columns', $response, 604800); //3600*24*7 = 604800 - 7 days

            //TODO: Test when success and $response = null!!!!
            return $this->success(null, $response);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function GetMetaBoxes()
    {
        try {
            ini_set('display_errors', 0);
            set_time_limit(0);

            $forceReload = $this->data['forceReload'];

            if( $forceReload !== 'true' ){
                $metaBoxesCached = get_transient('cusmin-metaboxes');
                if($metaBoxesCached) {
                    return $this->success(null, $metaBoxesCached);
                }
            }

            $postTypes = $this->data['postTypes'];
            $data = array();
            foreach ((array) $postTypes as $type){
                $url = get_site_url() . '/wp-admin/post-new.php?post-type=' . $type. '&cusmin-action=cusmin-list-metaboxes';

                // Create a stream
                $opts = [
                    "http" => [
                        "method" => "GET",
                        "header" =>
                            "Accept-language: ".$_SERVER['HTTP_ACCEPT_LANGUAGE']."\r\n" .
                            "Cookie: ".$_SERVER['HTTP_COOKIE']."\r\n"
                    ]
                ];

                // DOCS: https://www.php.net/manual/en/function.stream-context-create.php
                $context = stream_context_create($opts);

                // Open the file using the HTTP headers set above
                // DOCS: https://www.php.net/manual/en/function.file-get-contents.php

                try{
                    $content = file_get_contents($url, false, $context);
                    $data[$type] = json_decode($content);
                }catch (\Exception $e){
                    //$data[$type] = 'error: ' .$e->getMessage();
                }
            }
            $response = ['time' => current_time('mysql'), 'metaboxes' => $data];
            set_transient('cusmin-metaboxes', $response, 604800); //3600*24*7 = 604800 - 7 days
            return $this->success(null, $response);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function ConfigurationsReorder() {
        $newOrder = $this->data['newOrder'];

        try {
            CusminOptions::reorderConfigurations($newOrder);
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function NoticesLoad()
    {
        try {
            return $this->success(null, CusminOptions::noticesLoad());
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function NoticesCreate()
    {
        $type = $this->data['type'];
        $value = $this->data['value'];
        $message = $this->data['message'];
        $subject = $this->data['subject'];
        $messageType = $this->data['messageType'];

        //print_r($this->data);die;

        //send an email to a particular user
        //TODO: add support for sending emails to all users in group
        if($messageType == 'email'){
            try{
                $user = get_user_by( 'id', $value );
                //print_r($user);
                if($user){
                    $headers = array('Content-Type: text/html; charset=UTF-8');
                    if(!wp_mail( $user->user_email, $subject, $message, $headers)){
                        throw new Exception('Email to '.$user->user_email.' was not sent. <br/>Please check email configuration settings on the server.');
                    }
                }else{
                    throw new Exception('Unable to find user');
                }
                $message = $subject.'<br/><br/>'.$message;
            }catch (Exception $e) {
               return $this->fail($e->getMessage(), $e->getCode());
            }
        }

        try {
            return $this->success(
                null,
                CusminOptions::noticesCreate(
                    $type,
                    $value,
                    $message,
                    $messageType
                ));
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Deleted by administrator
     */
    protected function NoticesDelete()
    {
        try {
            return $this->success(
                null,
                CusminOptions::noticesDelete($this->data['notice'])
            );
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function DashboardWidgetsOrdering()
    {
        try {
            $username = $this->data['username'];
            $applyToExisting = (bool) $this->data['applyToExisting'];
            $configurationSlug = $this->data['configurationSlug'];

            if(!empty($username)){
                $refUser = get_user_by('login', $username);

                if($refUser){
                    if($applyToExisting){ //applies ordering option to all matched existing users
                        $userDashboardOrdering = get_user_meta( $refUser->ID, 'meta-box-order_dashboard', true );
                        $applyTo = CusminOptions::getOneApplyTo(CusminOptions::getDefaultConfigurationSlug());

                        $allUsers = get_users(array(
                                'exclude' => array($refUser->ID),
                            )
                        );

                        $caps = !empty($applyTo['capabilities'])?$applyTo['capabilities']:[];
                        $groups = !empty($applyTo['groups'])?$applyTo['groups']:[];
                        $ids = !empty($applyTo['ids'])?$applyTo['ids']:[];

                        //return $this->success($allUsers);

                        foreach((array) $allUsers as $u){
                            $update = false;
                            if(in_array($u->ID, $ids)){
                                $update = true;
                            }

                            if(!$update){
                                foreach((array) $u->roles as $uRole){
                                    if(in_array($uRole, $groups)){
                                        $update = true;
                                        break;
                                    }
                                }
                            }

                            if(!$update){
                                foreach((array) $u->allcaps as $capName => $calEnabled){
                                    if($calEnabled && in_array($capName, $caps)){
                                        $update = true;
                                        break;
                                    }
                                }
                            }

                            if($update){
                                try{
                                    update_user_meta( $u->ID, 'meta-box-order_dashboard', $userDashboardOrdering );
                                }catch (\Exception $e){}
                            }
                        }
                    }
                }else{
                    return $this->fail('User with username "'.$username.'" was not found. Please try with different username.');
                }
            }
            $message = 'Reference user "'.$username.'" was successfully set. New users will now have the same default dashboard widgets ordering as  "'.$username.'". ';
            if($applyToExisting){
                $message .= 'All existing matched users now have the same dashboard widgets ordering as the reference user.';
            }
            if(empty($username)){
                $message = 'The default dashboard ordering options was cleared.';
            }
            return $this->success(
                $message,
                CusminOptions::saveDashboardWidgetsUsername($configurationSlug, $username)
            );
        } catch (Exception $e) {
            return $this->fail($e->getMessage(), $e->getCode());
        }
    }

    protected function CustomizationsApplicationMode(){
        try{
            $mode = $this->data['mode'];
            $message = 'Mode was changed successfully.';
            return $this->success(
                $message,
                CusminOptions::changeCustomizationsApplicationMode($mode)
            );
        }catch (Exception $e){
            return $this->fail($e->getMessage() . $e->getTraceAsString(), $e->getCode());
        }
    }

    protected function loadAGCASettings(){
        $agcaExporter = new AGCAExporter();

        $settings = $agcaExporter->getSettings();

        if(!$settings){
            return $this->fail('AGCA is not activated.');
        }

        return $this->success(
            'AGCA Settings',
            $settings
        );
    }

    protected function apiTest(){
        try {
            require_once plugin_dir_path(dirname(__FILE__)) . '../packages/firebase/php-jwt/src/class-before-valid-exception.php';
            require_once plugin_dir_path(dirname(__FILE__)) . '../packages/firebase/php-jwt/src/class-expired-exception.php';
            require_once plugin_dir_path(dirname(__FILE__)) . '../packages/firebase/php-jwt/src/class-signature-invalid-exception.php';
            require_once plugin_dir_path(dirname(__FILE__)) . '../packages/firebase/php-jwt/src/class-jwk.php';
            require_once plugin_dir_path(dirname(__FILE__)) . '../packages/firebase/php-jwt/src/class-jwt.php';

            if(!defined('CUSMIN_API_SECRET') || CUSMIN_API_SECRET === '') {
                return $this->fail('CUSMIN_API_SECRET is not set in wp-config');
            }

            $jwt = JWT::encode(['data'=>'data'], 'afsafsfs');

            $decode = JWT::decode($jwt, CUSMIN_API_SECRET, array('HS256'));
            return $this->success(null, [
                'yes' => true,
                'jwt' => $decode
            ]);
        } catch (Exception $e) {
            return $this->fail($e->getMessage() , $e->getCode());
        }
    }


    /**
     * Resets all WP data to defaults
     */
    protected function resetAll()
    {
        try {
            CusminOptions::resetAll();
            return $this->success(null, null);
        } catch (Exception $e) {
            return $this->fail($e->getMessage() . $e->getTraceAsString(), $e->getCode());
        }
    }


    protected function success($message = '', $data = '')
    {
        $response = array(
            'success' => true,
            'message' => $message,
            'data' => $data
        );
        return $this->response($response);
    }

    protected function fail($message = '', $code = 0)
    {
        $response = array(
            'success' => false,
            'code' => $code,
            'message' => $message
        );
        return $this->response($response);
    }

    protected function response($response)
    {
        return json_encode($response);
    }

    /**
     * @throws Exception
     */
    protected function mustBeSlugFormat($string, $additionalMessage = ''){
        if(!$this->isSlugFormat($string)){
            throw new Exception('Incorrect string format. ' . $additionalMessage, 400);
        }
    }

    protected function isSlugFormat($string){
        return preg_match('/^[\w-]+$/', $string);

    }
}