<?php

class CusminOptions
{

    const OPTION_NAME = 'cusmin';
    const CONFIGURATIONS = 'configurations';
    const NOTICES = 'notices';
    const LOGIN = 'login';
    const LOGIN_SEL_CONFIG = 'selected-configuration';
    const CUSTOMIZATION = 'customization';
    const RULES = 'rules';
    const ACCESS_BY = 'access-by';
    const APPLY_TO = 'apply-to';
    //referenced user for the default dashboard options
    const DASHBOARD_WIDGETS_USERNAME = 'dashboard-widgets-username';
    const CUSTOMIZATIONS_APPLICATION_MODE = 'customizations-mode';
    const TRIAL_ENDED = 'trialEnded';
    const ADMIN_MENU_OPTION_ID = 71;
    private static $options;
    /**
     * @var CusminABApplier
     */
    public static $ab;

    //TODO: Move admin menu related methods
    /**
     * @var CusminAMApplier
     */
    private static $am;
    private static $originalTopMenu;
    private static $originalSubMenu;
    private static $originalDashboardWidgets;

    /**
     * @var CusminColumnsApplier
     */
    public static $columnsApplier;

    public function __construct()
    {
        self::$columnsApplier = new CusminColumnsApplier(self::$options);
    }


    private static function defaultValue()
    {
        return array(
            'default' => 'default', //default configuration name that is selected
            self::LOGIN => array(
                self::LOGIN_SEL_CONFIG => 'default'
            ),
            self::CONFIGURATIONS => array(
                'default' => self::newEmptyConfiguration('default', 'Default')
            )
        );
    }

    public static function get($force = false)
    {
        if ($force) {
            self::$options = get_option(self::OPTION_NAME);
            return self::$options;
        }
        if (self::$options == null) {
            self::$options = get_option(self::OPTION_NAME);
        }
        if (!self::$options) {
            return self::init();
        }

        return self::$options;
    }

    public static function save($options = null)
    {
        if ($options != null) {
            $options['id'] = uniqid();
            self::$options = $options;
        }
        update_option(self::OPTION_NAME, $options);
    }

    private static function init()
    {
        update_option(self::OPTION_NAME, self::defaultValue());
        self::$options = get_option(self::OPTION_NAME);
        return self::$options;
    }

    /**
     * Clears data from option, but does not delete it
     */
    public static function clear()
    {
        self::init();
    }

    /**
     * Deletes the option completely
     */
    public static function delete()
    {
        delete_option(self::OPTION_NAME);
    }

    //CONFIGURATIONS

    /**
     * @param $name friendly name
     * @return bool
     * @throws Exception
     */
    public static function addConfiguration($slug, $name)
    {
        $options = self::get();
        if ($slug == '') {
            throw new Exception('Invalid configuration name.');
        }
        if (isset($options[self::CONFIGURATIONS][$slug])) {
            //If renamed configuration
            if($options[self::CONFIGURATIONS][$slug]['name'] !== $name){
                $options[self::CONFIGURATIONS][$slug]['name'] = $name;
                self::save($options);
            }
            //Just use existing
            return true;
        }
        //New configuration
        $options[self::CONFIGURATIONS][$slug] = self::newEmptyConfiguration($slug, $name);
        self::save($options);
        return true;
    }

    public static function removeConfiguration($slug)
    {
        $options = self::get();
        self::validateConfigurationExists($slug);

        if ($slug == 'default') {
            throw new Exception('Default configuration can NOT be removed.');
        }

        unset($options[self::CONFIGURATIONS][$slug]);
        self::save($options);
        return true;
    }

    public static function updateConfigurationName($slug, $newName)
    {
        $options = self::get();
        $newSlug = sanitize_key($newName);
        self::validateConfigurationExists($slug);
        if (isset($options[self::CONFIGURATIONS][$newSlug])) {
            throw new Exception('Configuration with the same name already exists. Please choose different name.');
        }
        $options[self::CONFIGURATIONS][$newSlug] = $options[self::CONFIGURATIONS][$slug];
        unset($options[self::CONFIGURATIONS][$slug]);
        self::save($options);
        return true;
    }

    /**
     * Gets the name of the selected configuration
     * @return mixed
     */
    //TODO: Rename to selected configuration
    public static function getDefaultConfigurationSlug()
    {
        $options = self::get();
        $slug = $options['default'] == '' ? 'default' : $options['default'];

        //If selected configuration is removed somehow, use default
        if (!isset($options[self::CONFIGURATIONS][$slug])) {
            return 'default';
        }
        return $slug;
    }

    public static function getDefaultConfiguration(){
        return self::getOneConfigCustomization(self::getDefaultConfigurationSlug());
    }

    /**
     * Sets the name of the selected configuration
     * @param $slug - New default configuration name
     * @return bool
     * @throws Exception
     */
    public static function setDefaultConfigurationName($slug, $name)
    {
        $options = self::get();
        self::createConfigurationIfDoesntExists($slug, $name);
        $options['default'] = $slug;
        self::save($options);
        return true;
    }

    public static function getApplyToUsers()
    {
        $users = array();
        $configRules = self::getAllRules();

        foreach ((array) $configRules as $rule) {
            if(isset($rule['apply-to']['ids'])){
                $usrs = get_users([
                    'include' => $rule['apply-to']['ids'],
                    'fields'   => ['ID', 'user_email', 'display_name', 'user_url'],
                ]);
                $users = array_merge($users, $usrs);
            }
        }

        return (object) $users;
    }

    public static function toggleTrial($toggle)
    {
        $options = self::get();
        $options['trialEnded'] = $toggle;
        self::save($options);
        return true;
    }

    public static function setLoginSelectedConfiguration($slug, $name)
    {
        self::createConfigurationIfDoesntExists($slug, $name);

        $options = self::get();
        $options[self::LOGIN][self::LOGIN_SEL_CONFIG] = $slug;
        self::save($options);
        return true;
    }

    //TODO: Rename to public facing
    public static function getSelectedPublicFacingConfigurationSlug()
    {
        $options = self::get();
        $slug = $options[self::LOGIN][self::LOGIN_SEL_CONFIG] ? $options[self::LOGIN][self::LOGIN_SEL_CONFIG] : 'default';

        //Validate if configuration exists if not default
        if ($slug !== 'default') {
            try {
                self::validateConfigurationExists($slug);
            } catch (\Exception $e) {
                //Return default if selected configuration was not found
                return 'default';
            }
        }
        return $slug;
    }

    public static function getPublicFacingConfiguration()
    {
        return self::getOneConfigCustomization(self::getSelectedPublicFacingConfigurationSlug());
    }

    /**
     * Returns one configuration by provided slug
     * @param $slug
     * @return mixed
     * @throws Exception
     */
    public static function getOneConfiguration($slug)
    {
        $options = self::get();
        self::validateConfigurationExists($slug);
        return $options[self::CONFIGURATIONS][$slug];
    }

    public static function getOneConfigCustomization($slug)
    {
        $config = self::getOneConfiguration($slug);
        if (isset($config[self::CUSTOMIZATION])) {
            return $config[self::CUSTOMIZATION];
        } else {

            return array(
                'js' => '',
                'css' => '',
                'custom_css' => '',
                'custom_js' => '',
                'site_css' => '',
                'colorizer' => '',
                'php' => array(),
            );
        }
    }

    /**
     * Returns customizations from currently selected configuration
     * @return array
     */
    public static function getSelectedCustomizations()
    {
        return self::getOneConfigCustomization(
            self::getDefaultConfigurationSlug()
        );
    }

    /**
     * Returns all configurations from database
     * @return mixed
     */
    public static function getAllConfigurations()
    {
        $options = self::get();
        return $options[self::CONFIGURATIONS];
    }

    /**
     * Returns the list of configurations with rules only
     * @return array
     */
    public static function getAllRules()
    {
        $options = self::get();
        $rules = array();
        foreach ($options[self::CONFIGURATIONS] as $name => $c) {
            $cr = isset($c['rules']) ? $c['rules'] : null;
            //Filter out null/empty values
            if(isset($cr['apply-to']['ids'])) {
                $cr['apply-to']['ids'] = array_filter($cr['apply-to']['ids'], 'strlen');
            }
            $rules[$name] = $cr;
        }
        return $rules;
    }

    public static function getCustomDashboardWidgets()
    {
        $options = self::get();
        $widgets = array();
        try {
            $c = $options[self::CONFIGURATIONS][CusminOptions::getDefaultConfigurationSlug()][self::CUSTOMIZATION];
            if (isset($c['php'][72])) {
                $widgets = $c['php'][72];
            }
        } catch (\Exception $e) {
        }
        return $widgets;
    }


    public static function getAllConfigurationNames()
    {
        $options = self::get();
        $configs = array();

        foreach ($options[self::CONFIGURATIONS] as $name => $c) {
            $configs[$name] = isset($c['name']) ? $c['name'] : null;
        }
        return $configs;
    }

    public static function getCustomizationsMatrix()
    {
        $options = self::get();
        $customizations = array();
        if (isset($options[self::CONFIGURATIONS])) {
            $customizations = $options[self::CONFIGURATIONS];
            foreach ($options[self::CONFIGURATIONS] as $slug => $c) {
                $applyTo = isset($c['rules']['apply-to']) ? $c['rules']['apply-to'] : array();
                $customizations[$slug] = array(
                    'name' => isset($c['name']) ? $c['name'] : null,
                    'groups' => isset($applyTo['groups']) ? $applyTo['groups'] : array(),
                    'capabilities' => isset($applyTo['capabilities']) ? $applyTo['capabilities'] : array(),
                    'ids' => isset($applyTo['ids']) ? $applyTo['ids'] : array(),
                );
            }
        }
        return $customizations;
    }

    /*
     * Customizations applied to current user
     */
    public static function getAdminMenuUserCustomizations()
    {
        $customization = self::getCurrentUserCustomizations();
        $amOptions = null;
        if (isset($customization['php'][self::ADMIN_MENU_OPTION_ID])) {
            $amOptions = $customization['php'][self::ADMIN_MENU_OPTION_ID];
        }
        return $amOptions;
    }

    public static function getAdminMenuCustomizations()
    {   //TODO: Check what happens when applying admin menu customization. THis loads selected config, but we should check php and current user. Check if there's another method for that
        //TODO: Make sure that we call DB only once, not multiple times, not again just for admin menu
        $customization = self::getOneConfigCustomization(
            self::getDefaultConfigurationSlug()
        );
        $amOptions = null;
        if (isset($customization['php'][self::ADMIN_MENU_OPTION_ID])) {
            $amOptions = $customization['php'][self::ADMIN_MENU_OPTION_ID];
        }

        return $amOptions;
    }

    public static function getAdminBarCustomizations()
    {
        $ab = new CusminABApplier(new self);
        return $ab->getABCustomizations(self::getSelectedCustomizations());
    }

    public static function getAdminBarUserCustomizations()
    {
        $customization = self::getCurrentUserCustomizations();
        return CusminABApplier::getABCustomizations($customization);
    }

    public static function getAdminBarPublicCustomizations()
    {
        $customization = self::getPublicCustomizations();
        return CusminABApplier::getABCustomizations($customization);
    }

    public static function getCurrentUserNotices()
    {
        $options = self::$options;
        $userNotices = array();
        if (isset($options['notices'])) {
            $notices = $options['notices'];
            if (is_array($notices)) {
                foreach ($notices as $key => $n) {
                    if(isset($n['message-type']) && $n['message-type'] == 'email') continue; //don't show email notifications
                    $userId = get_current_user_id();
                    $value = $n['value'];
                    $conditionSet = false;
                    if ($n['type'] == 'groups') {
                        global $current_user;
                        $user_roles = $current_user->roles;
                        if (array_shift($user_roles) == $value) $conditionSet = true;
                    } else {//users
                        if ($value == $userId) $conditionSet = true;
                    }

                    //check if user dismissed notice
                    if ($conditionSet && !in_array($userId, $n['dismiss'])) {
                        $userNotices[$key] = $n;
                    }
                }
            }
        }
        return $userNotices;
    }

    //TODO: Similar function for Columns, make it more general
    public static function getCurrentUserHiddenMetaboxes()
    {
        $customizations = self::getCurrentUserCustomizations();
        if (isset($customizations['php']['133'])) {
            return $customizations['php']['133'];
        }
    }

    //CUSTOMIZATIONS
    /**
     * Updating customization in configuration
     * @param $configurationSlug
     * @param $customizations
     * @return bool
     * @throws Exception
     */
    public static function saveCustomization($configurationSlug, $configurationName, $customizations)
    {
        self::createConfigurationIfDoesntExists($configurationSlug, $configurationName);
        $options = self::get();
        $options[self::CONFIGURATIONS][$configurationSlug][self::CUSTOMIZATION] = $customizations;
        self::save($options);
        return true;
    }

    /**
     * Removing customization from configuration
     * @param $configurationSlug
     * @return bool
     * @throws Exception
     */
    public static function clearCustomization($configurationSlug)
    {
        self::validateConfigurationExists($configurationSlug);
        $options = self::get();
        $options[self::CONFIGURATIONS][$configurationSlug][self::CUSTOMIZATION] = array();
        //$options[self::CONFIGURATIONS][$configurationSlug][self::RULES][self::ACCESS_BY] = self::newAccessByRule();
        $options[self::CONFIGURATIONS][$configurationSlug][self::RULES][self::APPLY_TO] = self::newApplyToRule();
        self::save($options);
        return true;
    }

    /**
     * Clears post meta options, table columns and meta boxes
     */
    public static function clearPostMetaAppliedCustomizations(){
        $options = self::get();
        if (isset($options[self::CONFIGURATIONS])) {
            foreach ($options[self::CONFIGURATIONS] as $slug => $c) {
                try{
                    //options for hiding table columns
                    unset($options[self::CONFIGURATIONS][$slug][self::CUSTOMIZATION]['php'][93]);
                    //options for hiding meta boxes
                    unset($options[self::CONFIGURATIONS][$slug][self::CUSTOMIZATION]['php'][133]);
                }catch (\Exception $e){
                    echo $e->getMessage();
                    die;
                }
            }
        }
        self::save($options);
        return 'ok';
    }

    public static function removeCustomization($configurationSlug)
    {
        self::validateConfigurationExists($configurationSlug);
        $options = self::get();
        unset($options[self::CONFIGURATIONS][$configurationSlug]);
        if ($configurationSlug == 'default') {
            $options[self::CONFIGURATIONS]['default'] = self::newEmptyConfiguration('default', 'Default');
        }
        self::save($options);
        return true;
    }

    /**
     * Saves access by rules in specific configuration
     * @param $configurationSlug
     * @param $data
     * @return bool
     * @throws Exception
     */
    public static function saveAccessBy($configurationSlug, $data)
    {
        self::validateConfigurationExists($configurationSlug);

        //Make sure that current user is always admin
        $currentUserId = get_current_user_id();
        if (!isset($data['ids'][$currentUserId])) {
            $data['ids'][] = $currentUserId;
        }

        $options = self::get();
        $options[self::CONFIGURATIONS][$configurationSlug][self::RULES][self::ACCESS_BY] = $data;
        self::save($options);
        return true;
    }

    public static function resetAccessBy($configurationSlug)
    {
        self::validateConfigurationExists($configurationSlug);
        $options = self::get();
        $options[self::CONFIGURATIONS][$configurationSlug][self::RULES][self::ACCESS_BY] = self::newAccessByRule();
        self::save($options);
        return true;
    }

    /**
     * Saves apply to rules for specific configuration
     * @param $configurationSlug
     * @param $data
     * @return bool
     * @throws Exception
     */
    public static function saveApplyTo($configurationSlug, $configurationName, $data)
    {
        self::createConfigurationIfDoesntExists($configurationSlug, $configurationName);
        $options = self::get();
        $options[self::CONFIGURATIONS][$configurationSlug][self::RULES][self::APPLY_TO] = $data;
        self::save($options);
        return true;
    }

    public static function resetApplyTo($configurationSlug)
    {
        self::validateConfigurationExists($configurationSlug);
        $options = self::get();
        $options[self::CONFIGURATIONS][$configurationSlug][self::RULES][self::APPLY_TO] = self::newApplyToRule();
        self::save($options);
        return true;
    }

    public static function getOneApplyTo($configurationSlug, $configurationName = NULL)
    {
        self::createConfigurationIfDoesntExists($configurationSlug, $configurationName);
        $options = self::get();
        return isset($options[self::CONFIGURATIONS][$configurationSlug][self::RULES][self::APPLY_TO])
            ? $options[self::CONFIGURATIONS][$configurationSlug][self::RULES][self::APPLY_TO] : null;
    }

    public static function applyToSearchUsers($search = '')
    {
        $blogusers = get_users(['number' => 10, 'search' => '*'.$search.'*']);
        $users = array();

        foreach ($blogusers as $user) {
            $users[$user->ID] = array(
                'id' => $user->ID,
                'name' => $user->display_name . ' (' . $user->user_email . ')',
                'display_name' => $user->display_name,
                'email' => $user->user_email,
                'value' => $user->ID,
            );
        }

        return $users;
    }

    public static function reorderConfigurations($newOrder) {
        $configurations = self::getAllConfigurations();

        //Cleanup, remove any proposed slugs that are actually not existing in current set of local customizations
        $cleanedNewOrder = array();
        foreach ((array) $newOrder as $slug) {
            if(isset($configurations[$slug])) $cleanedNewOrder[] = $slug;
        }
        $newConfigurationsOrder =  array_replace(array_flip((array) $cleanedNewOrder), $configurations);

        $options = self::get();
        $options[self::CONFIGURATIONS] = $newConfigurationsOrder;
        self::save($options);
    }

    /**
     * Load all notices by admin
     */
    public static function noticesLoad()
    {
        $options = self::get();
        $notices = isset($options[self::NOTICES])?$options[self::NOTICES]:[];
        if (!is_array($notices)) {
            return array();
        }
        return $notices;
    }

    public static function noticesDismiss($noticeId)
    {
        $userId = get_current_user_id();
        $options = self::get();
        $notices = $options[self::NOTICES];
        if (is_array($notices)) {
            $notice = $notices[$noticeId];
            if (isset($notice) && is_array($notice['dismiss'])) {
                if (!in_array($userId, $notice['dismiss'])) {
                    $notice['dismiss'][] = $userId;
                    $options[self::NOTICES][$noticeId] = $notice;
                    self::save($options);
                }
            }
        }
    }

    /**
     * Delete by admin
     * @param $noticeId
     * @internal param $notice
     * @return bool
     */
    public static function noticesDelete($noticeId)
    {
        $options = self::get();
        $notices = $options[self::NOTICES];
        if (is_array($notices)) {
            unset($notices[$noticeId]);
        }
        $options[self::NOTICES] = $notices;
        self::save($options);
        return true;
    }

    //TODO: Future implementation to update notices
    //noticesUpdate

    /**
     * @param $type - user/group
     * @param $value
     * @param $message
     * @param $messageType
     * @return int
     */
    public static function noticesCreate($type, $value, $message, $messageType)
    {
        $options = self::get();
        $notices = $options[self::NOTICES];
        if (!$notices) {
            $notices = array();
        }
        $notices[] = array(
            'type' => $type,
            'value' => $value,
            'message' => $message,
            'message-type' => $messageType,
            'dismiss' => array() //the list of user ids that have dismissed the notification
        );
        $options[self::NOTICES] = $notices;
        self::save($options);
        //returns index of saved notice
        return sizeof($notices) - 1;
    }

    public static function isApplicableTo($configurationSlug, $current_user)
    {
        $userID = $current_user->ID;
        $userCaps = $current_user->allcaps;
        $userRoles = $current_user->roles;

        $applyTo = self::getOneApplyTo($configurationSlug);
        if (isset($applyTo['ids']) && is_array($applyTo['ids'])) {
            if (in_array($userID, $applyTo['ids'])) return true;
        }

        if (isset($applyTo['capabilities']) && is_array($applyTo['capabilities'])) {
            foreach ($applyTo['capabilities'] as $applyToCap) {
                foreach ($userCaps as $userCap) {
                    if ($applyToCap == $userCap) return true;
                }
            }
        }

        if (isset($applyTo['groups']) && is_array($applyTo['groups'])) {
            foreach ($applyTo['groups'] as $applyToGroup) {
                foreach ($userRoles as $userGroup) {
                    if ($applyToGroup == $userGroup) return true;
                }
            }
        }
        return false;
    }


    public static function saveDashboardWidgetsUsername($configurationSlug = null, $username = null)
    {
        $options = self::get();

        self::validateConfigurationExists($configurationSlug);

        if (empty($username)) {
            unset($options[self::CONFIGURATIONS][$configurationSlug][self::DASHBOARD_WIDGETS_USERNAME]);
        }else{
            $options[self::CONFIGURATIONS][$configurationSlug][self::DASHBOARD_WIDGETS_USERNAME] = $username;
        }
        self::save($options);
        return true;
    }

    public static function changeCustomizationsApplicationMode($mode = null)
    {
        $options = self::get();

        $options[self::CUSTOMIZATIONS_APPLICATION_MODE] = $mode;

        self::save($options);
        return true;
    }

    public static function getDashboardWidgetsUsername(){
        $options = self::get();
        $selConfig = self::getDefaultConfigurationSlug();
        if(!empty($options[self::CONFIGURATIONS][$selConfig][self::DASHBOARD_WIDGETS_USERNAME])){
            return $options[self::CONFIGURATIONS][$selConfig][self::DASHBOARD_WIDGETS_USERNAME];
        }
        return '';
    }

    public static function getCustomizationsApplicationMode(){
        $options = self::get();
        if(!empty($options[self::CUSTOMIZATIONS_APPLICATION_MODE])){
            return $options[self::CUSTOMIZATIONS_APPLICATION_MODE];
        }
        return CUSMIN_DEFAULT_CUSTOMIZATIONS_APPLICATION_MODE;
    }

    public static function getTrialEnded(){
        $options = self::get();
        if(!empty($options[self::TRIAL_ENDED])){
            return $options[self::TRIAL_ENDED];
        }
        return false;
    }

    public static function isFirstTimeLoadWithAGCAEnabled(){
        if(!is_admin()){
            return false;
        }
        $options = self::get();
        if(isset($options['id'])){
            return false;
        }
        return is_plugin_active('ag-custom-admin/plugin.php');
    }

    public static function resetAll()
    {
        $options = self::get();
        $options[self::CONFIGURATIONS] = array();
        $options[self::CONFIGURATIONS]['default'] = self::newEmptyConfiguration('default', 'Default');
        self::save($options);
        return true;
    }

    public static function getCurrentUserCustomizations()
    {
        return self::getUserCustomizations(wp_get_current_user());
    }

    public static function getCurrentUserColumnsCustomizations()
    {
        $customizations = self::getCurrentUserCustomizations();
        if (isset($customizations['php']['93'])) {
            return $customizations['php']['93'];
        }
    }

    /**
     * Returns all customizations that are applicable to current user
     * @param $current_user
     * @return array
     */
    public static function getUserCustomizations($current_user)
    {

        $options = self::get();
        $customizations = array(
            'js' => '',
            'css' => '',
            'php' => array(),
            'colorizer' => '',
            'custom_css' => '',
            'custom_js' => ''
        );

        if (isset($customizations['php']['93'])) {
            return $customizations['php']['93'];
        }

        $multiMatchingMode = isset($options[self::CUSTOMIZATIONS_APPLICATION_MODE])?$options[self::CUSTOMIZATIONS_APPLICATION_MODE]:'all'; //all, first, last
        $isCDebugMode = isset($_GET['cdebug']);
        if($isCDebugMode){echo"cdebug";print_r($multiMatchingMode);}
        if (isset($options[self::CONFIGURATIONS])) {
            foreach ($options[self::CONFIGURATIONS] as $slug => $c) {
                if (self::isApplicableTo($slug, $current_user)) {
                    $cust = self::getOneConfigCustomization($slug);
                    if($multiMatchingMode == 'first' || $multiMatchingMode == 'last'){
                        $customizations['css'] = isset($cust['css']) ? $cust['css'] : '';
                        $customizations['js'] = isset($cust['js']) ? $cust['js'] : '';
                        $customizations['php'] = isset($cust['php']) ? $cust['php'] : '';
                        $customizations['colorizer'] = isset($cust['colorizer']) ? $cust['colorizer'] : '';
                        $customizations['custom_css'] = isset($cust['custom_css']) ? $cust['custom_css'] : '';
                        $customizations['custom_js'] = isset($cust['custom_js']) ? $cust['custom_js'] : '';
                        if($multiMatchingMode == 'first'){
                            break;
                        }
                    }else{//all
                        $customizations['css'] .= isset($cust['css']) ? $cust['css'] : '';
                        $customizations['js'] .= isset($cust['js']) ? $cust['js'] : '';

                        //TODO: what if empty settings comes in the next one?
                        $customizations['php'] = isset($cust['php']) ? $cust['php'] : '';

                        //TODO: Merge all Colorizer settings
                        $customizations['colorizer'] = isset($cust['colorizer']) ? $cust['colorizer'] : '';
                        $customizations['custom_css'] .= isset($cust['custom_css']) ? $cust['custom_css'] : '';
                        $customizations['custom_js'] .= isset($cust['custom_js']) ? $cust['custom_js'] : '';
                    }
                }
            }
        }
        return $customizations;
    }

    /**
     * Returns all configurations matched with provided user
     * @param $user
     * @return array
     */
    public static function getUserConfigurations($user){
        $options = self::get();
        $configs = array();
        if (isset($options[self::CONFIGURATIONS])) {
            foreach ($options[self::CONFIGURATIONS] as $slug => $c) {
                if (self::isApplicableTo($slug, $user)) {
                    $configs[]=$c;
                }
            }
        }
        return $configs;
    }

    /**
     * Returns login page customizations
     */
    //TODO: Check if we need this after loading directly
    public static function getLoginCustomizations()
    {
        $cust = self::getOneConfigCustomization(self::getSelectedPublicFacingConfigurationSlug());
        return array(
            'css' => isset($cust['login_css']) ? $cust['login_css'] : '',
            'js' => isset($cust['login_js']) ? $cust['login_js'] : '',
            'php' => isset($cust['php']) ? $cust['php'] : '',
        );
    }

    public static function getPublicCustomizations()
    {
        $cust = self::getOneConfigCustomization(self::getSelectedPublicFacingConfigurationSlug());
        return array(
            'css' => isset($cust['site_css']) ? $cust['site_css'] : '',
            'js' => isset($cust['site_js']) ? $cust['site_js'] : '',
            'php' => isset($cust['php']) ? $cust['php'] : ''
        );
    }

    public static function getId()
    {
        $options = self::get();
        return isset($options['id']) ? $options['id'] : '';
    }

    public static function isPublicCSSSet()
    {
        return !empty(self::getSiteCustomCSSCustomizations()) || self::abContainsPublicCustomizations();
    }

    /**
     * In case public customizations are set in admin bar, please enable cusmin JS script
     */
    protected static function abOneOfTypesContainsPublicCustomizations($abParentsOrChildren) {
        foreach ((array) $abParentsOrChildren as $item) {
            foreach((array) $item as $key => $value) {
                if(strpos($key, 'color') === 0 ||
                    $key === "target" ||
                    $key === "icons" ||
                    $key === "custom"
                ) {
                    //print_r([$key => $item ]);die;
                    return true;

                }
            }
        }
        return false;
    }

    protected static function abContainsPublicCustomizations() {
        //Check if AB customizations are set for front-end
        try {
            $customizations = is_user_logged_in() ? self::getCurrentUserCustomizations() : self::getPublicCustomizations();

            //If user is not logged in
            if(!is_user_logged_in()) {
                //If admin bar is disabled for logged out users (138 option to enable admin bar for non-logged in users)
                if(!(isset($customizations['php'][138]) && $customizations['php'][70] == 1)){
                    return false;
                }
            }

            if(isset($customizations['php'][70])) {
                $customizations = $customizations['php'][70];
                if((isset($customizations['parent'])
                    && self::abOneOfTypesContainsPublicCustomizations($customizations['parent'])) ||
                    (isset($customizations['child'])
                        && self::abOneOfTypesContainsPublicCustomizations($customizations['child']))) {
                    return true;
                }
            }
        }catch (\Exception $e){}
        return false;
    }

    public static function isPublicJSSet()
    {
        return !empty(self::getSiteCustomJSCustomizations()) || self::abContainsPublicCustomizations();
    }

    public static function isLoginCSSSet()
    {
        return !empty(self::getLoginCSSCustomizations());
    }

    public static function isLoginJSSet()
    {
        return !empty(self::getLoginJSCustomizations());
    }

    public static function getSiteCustomCSSCustomizations()
    {
        $cust = self::getPublicFacingConfiguration();

        $css = isset($cust['site_css']) ? $cust['site_css'] : '';

        $ca = new CusminApplier(isset($cust['php'])?$cust['php']:'');

        if($ca->setting_138()){
            $css .= CusminOptions::getTopABIconsColorsForSitePages();
        }
        return $css;
    }

    public static function getSiteCustomJSCustomizations()
    {
        $cust = self::getPublicFacingConfiguration();
        $js = isset($cust['site_js']) ? $cust['site_js'] : '';

        if(!empty($cust['php'])){
            $ca = new CusminApplier($cust['php']);

            if($ca->setting_138()){
                $js .= CusminOptions::getAdditionalJSScriptABIcons();
            }
        }

        return $js;
    }

    public static function getLoginCSSCustomizations()
    {
        $cust = self::getPublicFacingConfiguration();
        $css = 'body, html{height:100%;width:100%;display:table;}';

        if(!empty($cust['login_css'])) {
            return $css . $cust['login_css'];
        }

        return '';
    }

    public static function getLoginJSCustomizations()
    {
        $cust = self::getPublicFacingConfiguration();
        return isset($cust['login_js']) ? $cust['login_js'] : '';
    }

    public static function getAdminCSSCustomizations()
    {
        $cust = self::getCurrentUserCustomizations();
        $amIconsColorsCss = CusminOptions::getTopMenuColorsForCurrentUser();
        $abIconsColorsCss = CusminOptions::getTopABIconsColorsForCurrentUser();
        $css = isset($cust['css']) ? $cust['css'] : '';

        $ca = new CusminApplier($cust['php']);

        if($ca->setting_150()){
            $css.=' .cusmin .notice{display:none;}.cusmin .notice.cusmin-notice{display:block;}';
        }

        return $css . $amIconsColorsCss . $abIconsColorsCss;
    }

    public static function getCommonCss(){
        $css = '';

        if(is_admin_bar_showing()) {
            $css .='
@media screen and ( max-width: 782px ) {
    #wp-toolbar > ul > li.cusmin-ab-icon {
        display: block;
        position: static;
    }
    #wp-toolbar > ul > li.cusmin-ab-icon .ab-item{
        text-indent: 100%;
        white-space: nowrap;
        overflow: hidden;
        width: 52px;
        padding: 0;
        color: #a0a5aa;
        position: relative;
        transition:none;
    }
    #wp-toolbar > ul > li.cusmin-ab-icon:hover .ab-item,
    #wp-toolbar > ul > li.cusmin-ab-icon.hover .ab-item{
        text-indent: 0;
    }
    #wp-toolbar > ul > li.cusmin-ab-icon .ab-item:before{
        display: block;
        text-indent: 0;
        font: normal 32px/1 dashicons;
        speak: none;
        top: 3px !important;
        width: 52px;
        text-align: center;
        -webkit-font-smoothing: antialiased;
        position: relative;
        margin: 4px 0;
    }
    
    .cusmin #wpadminbar .cusmin-ab-icon .material-icons{
        margin-top: -10px !important;
        font-size: 30px !important;
        margin-left: -17px !important;
    }
}

        ';

        }
        return Cusmin::minifyCSS($css);
    }

    public static function getAdminAdditionalScripts()
    {
        //TODO: Include only when these options are actually set
        //or find better way to use them

        //For admin menu
        $additionalJS = '
        try{
            var faTopItems = document.getElementsByClassName("menu-top cusmin-fa");
            for(var i in faTopItems){
                if(faTopItems[i].getElementsByClassName && typeof faTopItems[i].className !== "undefined"){
                    var obj = faTopItems[i].className.split("cusmin-fa fa-")[1];
                    /*console.debug(obj);*/
                    if(obj){
                        var icon = obj.split(" ")[0];
                        var faClass = " fa fa-" + icon;
                        faTopItems[i].getElementsByClassName("wp-menu-image")[0].className +=faClass;
                    }
                }
            };

            var miTopItems = document.getElementsByClassName("menu-top cusmin-mi");
            for(var i in miTopItems){
                if(miTopItems[i].getElementsByClassName && typeof miTopItems[i].className !== "undefined"){
                    var obj = miTopItems[i].className.split("cusmin-mi mi-")[1];
                    /*console.debug(obj);*/
                    if(obj){
                        var icon = obj.split(" ")[0];
                        miTopItems[i].getElementsByClassName("wp-menu-image")[0].className +=" material-icons";
                        miTopItems[i].getElementsByClassName("wp-menu-image")[0].innerText =icon;
                    }
                }
            };
       }catch(e){console.log(e);}
        ';

        //admin menu, remove top item when capability is set, and top item is now empty
        $additionalJS .= '
            try{
                var adminmenu = document.getElementById("adminmenu");
                if(adminmenu){
                    var tops = document.getElementById("adminmenu").children;
                    for(var i in tops){
                    /*console.debug(tops[i].className);*/
                        if(tops[i].className && tops[i].className.indexOf("menu-top") !== -1 && tops[i].innerHTML == ""){
                            tops[i].remove();
                        }
                    }
                }
            }catch(e){console.log(e);}
        ';

        //AB Additional script
        if(is_user_logged_in()){
            //if it's not logged in, ab JS is already applied publicly
            $additionalJS .= self::getAdditionalJSScriptABIcons();
        }

        //Notices dismiss
        //TODO: Find better way to use off Cusmin page cusmin JS globals
        $additionalJS .= '
        try{
            if(window.cusminPublic){
                window.cusminPublic.dismissNotice = function(notice){
                     var xmlhttp = new XMLHttpRequest();
                     var url = window.location.origin + "/wp-admin/options-general.php?page=cusmin&cusmin-action=dismiss-notice&notice=" + notice;
                     xmlhttp.open("GET", url, true);
                     xmlhttp.send();
                };
            }
        }catch(e){console.log(e);};
        ';

        //TODO: Replace jQuery with Vanilla
        //TODO: Find out why this happens?
        //admin menu sub items - removes those that are empty
        $additionalJS .= '
         (function($){
                $(".wp-submenu > li > a:empty").remove();
            })(jQuery);
        ';

        //TODO: Replace jQuery with Vanilla
        //for dashboard RSS
        $additionalJS .= '
            (function($){
                $("#dashboard-widgets .rss-widget").parent().css({"max-height":"555px"});
            })(jQuery);
        ';

        //revert cusmin version
        $additionalJS .= '
            (function($){
                $("body a.cusmin-revert-version").click(function(e){
                    var $link = $(e.target);
                    if(!$link.hasClass("processing")){
                        $link.addClass("processing");
                        $.get("/wp-admin/options-general.php?page=cusmin&cusmin-action=revert-version", function(){
                            $link.addClass("done");
                            $link.trigger("click");
                        });
                        console.log("clci");
                        return false;
                    }else{
                        if($link.hasClass("done")){
                            $link.removeClass("done");
                            $link.removeClass("processing");
                        }else{
                            return false;
                        }
                    }
                });
            })(jQuery);
        ';

        //new target top menu items
        $additionalJS .= '
            (function($){
                $("a.cusmin-target-_blank, .cusmin-target-_blank > a").attr("target", "_blank")
            })(jQuery);
        ';

        //$additionalJS .= ' document.getElementsByTagName("html")[0].style.visibility = "visible";';

        /*
         *
         * $("#dashboard-widgets .controls a").on("click", function(e){
                    e.preventDefault();
                    var $this = $(this);
                    var $rss = $this.closest(".rss-widget");
                    var page = $rss.data("page");
                    var perPage = $rss.data("per-page");
                    var total = $rss.find("li").length;
                    var offset = (page - 1) * perPage + 1;
                    if($this.hasClass("next")){
                        page++;
                    }else{
                        page--;
                    }
                    $rss.find(".previous").show();
                    $rss.find(".next").show();
                    if(page == 0){
                       // $rss.find(".previous").hide();
                    }else if(total/perPage >= page - 1){
                      //  $rss.find(".next").hide();
                    }
                    $rss.find("li").addClass("hidden");
                    $rss.find("li").slice(offset, offset + perPage).removeClass("hidden");
                    $rss.data("page", page);

                });
         */

        //TODO: Create a setting
        /* $additionalJS.='
             jQuery(document).ready(function() {
                 clock = jQuery(".cusmin-flipclock").FlipClock({
                     clockFace: "TwelveHourClock"
                 });
             });
         ';*/
        return $additionalJS;
    }

    private static function getAdditionalJSScriptABIcons(){
        //For admin bar
        $additionalJS = '
        try{
            var parentItems = document.getElementsByClassName("cusmin-ab-icon");
            for(var i in parentItems){
                console.debug(parentItems[i].className);
                if(parentItems[i].getElementsByClassName && typeof parentItems[i].className !== "undefined"){
                    var $o = parentItems[i];
                    var cls = $o.className.split("cusmin-ab-icon ")[1];
                    if(cls){
                        var icon = "";
                        var context ="";
                        if(cls.indexOf("dashicons-") === 0){
                            context = "dashicons";
                        }else if(cls.indexOf("fa-") === 0){
                            context = "fa";
                        }else if(cls.indexOf("mi-") === 0){
                            context = "mi";
                        };

                        if(context == "mi"){
                            var target = $o.getElementsByClassName("ab-item")[0];
                            //target.className +=" material-icons";
                            var html = target.innerHTML;
                            cls = cls.split(" ")[0];
                            cls = cls.split("mi-")[1];
                            if(html.indexOf("material-icons") === -1){
                                target.innerHTML =  "<div class=\"material-icons\" style=\"font-family:\'Material Icons\';font-size: 22px;display:block;float:left;max-width: 60px;margin-right: 6px !important;margin-top:-10px;\">" + cls  + "</div>";
                                target.innerHTML += "<span class=\"material-icons-text\" style=\"\">"+html+"</span>";
                            }
                        }else{
                            var icon = context + " " + cls;
                            $o.getElementsByClassName("ab-item")[0].className +=" " + icon;
                        }
                    }
                    /*console.debug(obj);*/
                }
            };
            //Apply Cusmin target blank
            var list = document.getElementsByClassName("cusmin-target-_blank");
            for (var i = 0; i < list.length; i++) {
                list[i].setAttribute("target", "_blank");
                var children = list[i].children;
                for(var j = 0; j < children.length; j++) {
                    try{
                        if(children[j].tagName.toLowerCase() == "a"){
                            children[j].setAttribute("target", "_blank");
                        }
                    }catch(e){}
                }
            }
        }catch(e){console.log(e);}
        ';
        return $additionalJS;
    }

    public static function getAdminJSCustomizations()
    {
        $cust = self::getCurrentUserCustomizations();
        $ca = new CusminApplier($cust['php']);

        // $colorScheme = get_user_option( 'admin_color' );
        // global $admin_colors;

        //print_r($admin_colors);

        $script = $cust['js']
            . $ca->setting_19()
            . $ca->setting_11()
            . $ca->setting_69();


        //TODO: Enable this script for tag replacing
        //Replace Cusmin placeholders with anything we need, wrap or update code
        //It has to be backward compatible

        //$script = self::cusminTagsReplacement($script);

        return $script;
    }

    //Replace Cusmin tags with some WordPress functions like shortcodes
    /**
     * @param string $text
     * @return string
     */
    private static function cusminTagsReplacement($text = '')
    {

        //Replace shortcodes:
        preg_match('~<CUSMIN-SHORTCODE>([^{]*)</CUSMIN-SHORTCODE>~i', $text, $match);

        //echo has_shortcode($text, 'video');
        //echo shortcode_exists('videos');

        //Found short code:
        if (sizeof($match) > 0) {
            $textWithTags = $match[0];
            $innerText = $match[1];
            $text = str_replace($textWithTags, wp_video_shortcode($innerText), $text);
        }

        return $text;
    }

    public static function getTopMenuColorsForCurrentUser()
    {
        $customizations = self::getAdminMenuUserCustomizations();
        if (!$customizations) {
            return '';
        }
        if (!isset($customizations['top'])) {
            return '';
        }
        $ac = $customizations['top'];

        $css = '';
        foreach ($ac as $menu) {
            if (isset($menu['color']) && isset($menu['icon'])) {
                $color = self::hexColorNoHash($menu['color']);
                if ($color) {
                    $css .= '.cusmin #adminmenu li.menu-top.color-' . $color . ' .dashicons-before:before{color:' . $menu['color'] . ' !important}';
                    $css .= '.cusmin #adminmenu li.menu-top.color-' . $color . ' .material-icons{color:' . $menu['color'] . ' !important}';
                }
            }
            if (isset($menu['colorHover']) && isset($menu['icon'])) {
                $color = self::hexColorNoHash($menu['colorHover']);
                if ($color) {
                    $css .= '.cusmin #adminmenu li.menu-top.color-hover-' . $color . ':hover .dashicons-before:before{color:' . $menu['colorHover'] . ' !important}';
                    $css .= '.cusmin #adminmenu li.menu-top.color-hover-' . $color . ':hover .material-icons{color:' . $menu['colorHover'] . ' !important}';
                }
            }
            if (isset($menu['colorTxt'])) {
                $color = self::hexColorNoHash($menu['colorTxt']);
                if ($color) {
                    $css .= '.cusmin #adminmenu li.menu-top.color-txt-' . $color . ' > a{color:' . $menu['colorTxt'] . ' !important}';
                }
            }
            if (isset($menu['colorTxtHover'])) {
                $color = self::hexColorNoHash($menu['colorTxtHover']);
                if ($color) {
                    $css .= '.cusmin #adminmenu li.menu-top.color-txt-hover-' . $color . ':hover > a{color:' . $menu['colorTxtHover'] . ' !important}';
                }
            }
            if (isset($menu['colorBg'])) {
                $color = self::hexColorNoHash($menu['colorBg']);
                if ($color) {
                    $css .= '.cusmin #adminmenu li.menu-top.color-bg-' . $color . ' > a{background:' . $menu['colorBg'] . ' !important}';
                }
            }
            if (isset($menu['colorBgHover'])) {
                $color = self::hexColorNoHash($menu['colorBgHover']);
                if ($color) {
                    $css .= '.cusmin #adminmenu li.menu-top.color-bg-hover-' . $color . ':hover > a{background:' . $menu['colorBgHover'] . ' !important}';
                }
            }
        }
        return $css;
    }

    public static function getTopABIconsColorsForCurrentUser()
    {
        return self::buildABIconsColorsCSS(self::getAdminBarUserCustomizations());
    }

    public static function getTopABIconsColorsForSitePages(){
        return self::buildABIconsColorsCSS(self::getAdminBarPublicCustomizations());
    }

    private static function buildABIconsColorsCSS($customizations = ''){
        if (!$customizations) {
            return '';
        }
        if (!isset($customizations['parent'])) {
            return '';
        }
        $ac = $customizations['parent'];

        $css = '';
        foreach ($ac as $menu) {
            if (isset($menu['color'])) {
                $color = self::hexColorNoHash($menu['color']);
                $isTopSecondary = false;
                if ($color) {
                    if (isset($menu['id'])) {
                        if ($menu['id'] == 'top-secondary') {
                            $isTopSecondary = true;
                        }
                    }
                    if ($isTopSecondary) {
                        $css .= '.cusmin #wpadminbar #wp-admin-bar-top-secondary:before{color:' . $menu['color'] . ' !important}';
                    } else {
                        $css .= '.cusmin #wpadminbar .cusmin-ab-icon.color-' . $color . ' .ab-item:before,';
                        $css .= '.cusmin #wpadminbar .cusmin-ab-icon.color-' . $color . ' .ab-icon:before{color:' . $menu['color'] . ' !important}';
                        $css .= '.cusmin #wpadminbar .cusmin-ab-icon.color-' . $color . ' .material-icons{color:' . $menu['color'] . ' !important}';
                    }
                }
            }
            if (isset($menu['colorHover'])) {
                $colorHover = self::hexColorNoHash($menu['colorHover']);
                $isTopSecondary = false;
                if ($colorHover) {
                    if (isset($menu['id'])) {
                        if ($menu['id'] == 'top-secondary') {
                            $isTopSecondary = true;
                        }
                    }
                    if ($isTopSecondary) {
                        $css .= '.cusmin #wpadminbar #wp-admin-bar-top-secondary:before{color:' . $menu['colorHover'] . ' !important}';
                    } else {
                        $css .= '.cusmin #wpadminbar .cusmin-ab-icon.color-hover-' . $colorHover . ':hover .ab-item:before,';
                        $css .= '.cusmin #wpadminbar .cusmin-ab-icon.color-hover-' . $colorHover . ':hover .ab-icon:before{color:' . $menu['colorHover'] . ' !important}';
                        $css .= '.cusmin #wpadminbar .cusmin-ab-icon.color-hover-' . $colorHover . ':hover .material-icons{color:' . $menu['colorHover'] . ' !important}';
                    }
                }
            }
            if (isset($menu['colorTxt'])) {
                $colorTxt = self::hexColorNoHash($menu['colorTxt']);
                $isTopSecondary = false;
                if ($colorTxt) {
                    if (isset($menu['id'])) {
                        if ($menu['id'] == 'top-secondary') {
                            $isTopSecondary = true;
                        }
                    }
                    if ($isTopSecondary) {
                        //$css .= '.cusmin #wpadminbar #wp-admin-bar-top-secondary:before{color:' . $menu['colorTxt'] . ' !important}';
                    } else {
                        $css .= '.cusmin #wpadminbar .color-txt-' . $colorTxt . ' > a.ab-item,';
                        $css .= '.cusmin #wpadminbar .color-txt-' . $colorTxt . ' > a.ab-icon{color:' . $menu['colorTxt'] . ' !important}';

                        //and on hover set same color, because hover might not be set in colorTxtHover
                        $css .= '.cusmin #wpadminbar .cusmin-ab-icon.color-txt-hover-' . $colorTxt . ':hover > .ab-item,';
                        $css .= '.cusmin #wpadminbar .cusmin-ab-icon.color-txt-hover-' . $colorTxt . ':hover > .ab-icon{color:' . $menu['colorTxt'] . ' !important}';
                    }
                }
            }
            if (isset($menu['colorTxtHover'])) {
                $colorTxtHover = self::hexColorNoHash($menu['colorTxtHover']);
                $isTopSecondary = false;
                if ($colorTxtHover) {
                    if (isset($menu['id'])) {
                        if ($menu['id'] == 'top-secondary') {
                            $isTopSecondary = true;
                        }
                    }
                    if ($isTopSecondary) {
                        //$css .= '.cusmin #wpadminbar #wp-admin-bar-top-secondary:before{color:' . $menu['colorTxtHover'] . ' !important}';
                    } else {
                        $css .= '.cusmin #wpadminbar .cusmin-ab-icon.color-txt-hover-' . $colorTxtHover . ':hover > .ab-item,';
                        $css .= '.cusmin #wpadminbar .cusmin-ab-icon.color-txt-hover-' . $colorTxtHover . ':hover > .ab-icon{color:' . $menu['colorTxtHover'] . ' !important}';
                    }
                }
            }
            if (isset($menu['colorBg'])) {
                $colorBg = self::hexColorNoHash($menu['colorBg']);
                $isTopSecondary = false;
                if ($colorBg) {
                    if (isset($menu['id'])) {
                        if ($menu['id'] == 'top-secondary') {
                            $isTopSecondary = true;
                        }
                    }
                    if ($isTopSecondary) {
                        //$css .= '.cusmin #wpadminbar #wp-admin-bar-top-secondary:before{color:' . $menu['colorBg'] . ' !important}';
                    } else {
                        $css .= '.cusmin #wpadminbar .cusmin-ab-icon.color-bg-' . $colorBg . ',';
                        $css .= '.cusmin #wpadminbar .cusmin-ab-icon.color-bg-' . $colorBg . '{background:' . $menu['colorBg'] . ' !important; max-height: 32px;}';

                        $css .= '@media screen and (max-width: 782px){.cusmin #wpadminbar .cusmin-ab-icon.color-bg-' . $colorBg . '{max-height: 100%;}}';
                    }
                }
            }
            if (isset($menu['colorBgHover'])) {
                $colorBgHover = self::hexColorNoHash($menu['colorBgHover']);
                $isTopSecondary = false;
                if ($colorBgHover) {
                    if (isset($menu['id'])) {
                        if ($menu['id'] == 'top-secondary') {
                            $isTopSecondary = true;
                        }
                    }
                    if ($isTopSecondary) {
                        //$css .= '.cusmin #wpadminbar #wp-admin-bar-top-secondary:before{color:' . $menu['colorBgHover'] . ' !important}';
                    } else {
                        $css .= '.cusmin #wpadminbar .cusmin-ab-icon.color-bg-hover-' . $colorBgHover . ':hover,';
                        $css .= '.cusmin #wpadminbar .cusmin-ab-icon.color-bg-hover-' . $colorBgHover . ':hover > a{background:' . $menu['colorBgHover'] . ' !important; max-height: 32px;}';

                        $css .= '@media screen and (max-width: 782px){.cusmin #wpadminbar .cusmin-ab-icon.color-bg-hover-' . $colorBgHover . ':hover > a{ max-height: 100%;}}';
                    }
                }
            }
        }
        return $css;
    }

    /**
     * Returns hex color without starting hash
     * @param $color
     * @return mixed color without hash
     */
    private static function hexColorNoHash($color)
    {
        return preg_replace("/[^a-zA-Z0-9]+/", "", $color);
    }

    public static function getCustomMenuTop()
    {
        $updatedmenu_top = array();
        try{
            //TODO: make sure to call it only once for top and sub
            $ac = self::getAdminMenuCustomizations();
            if (!$ac) {
                return array();
            }
            $updatedmenu_top = $ac['top'];
        }catch (Exception $e){
            return array();
        }
        return $updatedmenu_top;
    }

    public static function getCustomMenuSub()
    {
        $updatedmenu_top = array();
        try{
            //TODO: make sure to call it only once for top and sub
            $ac = self::getAdminMenuCustomizations();
            if (!$ac) {
                return array();
            }
            $updatedmenu_sub = isset($ac['sub'])?$ac['sub']:[];
            //remove_submenu_page('themes.php','theme-editor.php'); // Remo
        }catch (Exception $e){
            return array();
        }
        return $updatedmenu_sub;
    }

    public static function getOriginalMenuTop()
    {
        $updatedMenu = array();

        foreach (self::$originalTopMenu as $index => $item) {
            //TODO: Check what to do with separators
            $url = $item[2];
            /* if(preg_match('/separator/', $url)){
                 continue;
             }*/
            $updatedMenu[] = $item;
        }
        return $updatedMenu;
    }

    public static function getOriginalMenuSub()
    {
        return self::$originalSubMenu;
    }

    public static function getOriginalAdminBarParentNodes()
    {
        return self::$ab->getOriginalParents();
    }

    public static function getCustomizedAdminBarParentNodes()
    {
        //TODO: make sure to call it only once for top and sub
        $abc = self::getAdminBarCustomizations();

        if (!$abc) {
            return array();
        }

        return isset($abc['parent']) ? $abc['parent'] : array();
    }

    public static function getCustomizedAdminBarChildNodes()
    {
        $abc = self::getAdminBarCustomizations();

        if (!$abc) {
            return array();
        }

        return isset($abc['child']) ? $abc['child'] : array();
    }

    public static function getEnabledPluginsList(){
        $plugins = array();

        if(is_plugin_active('buddypress/bp-loader.php')){
            $plugins[] = 'buddypress';
        }

        return $plugins;
    }

    //TODO: Move to separate file, everything related to admin menu
    public static function getMenuTop()
    {
        //global $menu;
        $menu = self::$originalTopMenu;
        $updatedMenu = array();

        //TODO: Get this from saved settings
        $hideSeparators = true;

        //remove_menu_page('tools.php'); // Remove the Tools menu

        $mapped_top_index = array();
        foreach ($menu as $index => $item) {
            $url = $item[2];
            /* if($hideSeparators and preg_match('/separator/', $url)){
                 continue;
             }*/
            $updatedMenu[] = $item;
            $mapped_top_index[$url] = $index;
        }
        // $newmenu_top = array();
        //$menu = $this->adminMenuApplyTop($newmenu_top);
        return $updatedMenu;
    }

    public static function getMenuSub()
    {
        //global $menu;
        //global $submenu;
        $submenu = self::$originalSubMenu;
        //$updatedMenu = array();

        $mapped_top_index = array();
        //$newmenu_sub = array();
        //$submenu = $this->adminMenuApplySub($newmenu_sub);
        return $submenu;
    }

    /**
     * Preserves original admin menu, before customizations
     */
    function preserveOriginalMenu()
    {
        global $menu;
        global $submenu;
        self::$originalTopMenu = $menu;
        self::$originalSubMenu = $submenu;
    }

    //TODO: Try to find a way to get wp meta boxes on non dashboard page,
    //if not possible, delete this function
    public static function preserveDashboardWidgetsList()
    {

        global $wp_meta_boxes;
        print_r($wp_meta_boxes);
        $widgets = array();
        if (isset($wp_meta_boxes['dashboard'])) {
            foreach ($wp_meta_boxes['dashboard']['normal']['core'] as $key => $w) {
                $widgets[$key] = array(
                    'title' => $w['title'],
                    'place' => 'normal'
                );
            }
            foreach ($wp_meta_boxes['dashboard']['side']['core'] as $key => $w) {
                $widgets[$key] = array(
                    'title' => $w['title'],
                    'place' => 'side'
                );
            }
            self::$originalDashboardWidgets = $widgets;
        }
    }

    public static function getColumnsApplier()
    {
        return self::$columnsApplier;
    }

    public function setAdminBar()
    {
        try {
            self::$ab = new CusminABApplier($this);
        } catch (\Exception $e) {
        }
    }

    public function adminBarApplyUserCustomizations(){
        try {
            //Set original items
            //TODO: Can't we use this directly from AB helper class?
            //  self::$originalAdminBarParentNodes = $ab->getOriginalParents();
            // self::$originalAdminBarChildNodes = $ab->getOriginalChildren();

            //get AB customizations
            $customizations = self::getCurrentUserCustomizations();
            self::$ab->applyCustomizations($customizations);
        } catch (\Exception $e) {
            //print_r($e);die;
        }
    }

    public function adminBarApplyPublicCustomizations(){
        try {
            //get AB customizations
            $customizations = self::getPublicCustomizations();
            $ca = new CusminApplier($customizations['php']);
            if($ca->setting_138()){
                $frontEndOptions = $ca->setting_134();
                if(is_array($frontEndOptions) && in_array('vendor-buddypress-hidelogin', $frontEndOptions)){
                    global $wp_admin_bar;
                    $wp_admin_bar->remove_node('bp-login');
                }
            }
            self::$ab->applyCustomizations($customizations);
        } catch (\Exception $e) {
            cusmin_exception($e);
        }
    }

    /**
     * Removes all default button, except user's defined custom buttons
     */
    function adminMenuRemoveItems()
    {
        global $menu;
        foreach ($menu as $index => $item) {
            if (strpos($item[4], 'cusmin-top-item') == false && strpos($item[4], 'toplevel_page_cusmin_') == false) {
                remove_menu_page($item[2]);
            }
        }
    }

    private function colorClassFromColor($cItem, $colorName, $suffix = '')
    {
        if (isset($cItem[$colorName]) && $cItem[$colorName]) {
            $color = preg_replace("/[^a-zA-Z0-9]+/", "", $cItem[$colorName]);
            return $color ? ' color-' . ($suffix ? $suffix . '-' : '') . $color : '';
        }
        return '';
    }

    function adminMenuApplyTop()
    {
        //TODO: Check it creates objects and arrays in JS
        global $menu;

        $amCustomizations = $this->getAdminMenuUserCustomizations();
        if (isset($amCustomizations['top'])) {
            $amCustomizations = $amCustomizations['top'];
            $updatedMenu = array();
            $coveredURLs = array();

            foreach ($amCustomizations as $cIndex => $cItem) {
                $url = $cItem['url'];
                $coveredURLs[] = $url;

                if (isset($cItem['hidden']) && $cItem['hidden']) {
                    //skip
                    continue;
                }

                if (isset($cItem['custom']) && $cItem['custom']) {
                    $class = 'menu-top cusmin-top-item';
                    $class .= $this->colorClassFromColor($cItem, 'color');
                    $class .= $this->colorClassFromColor($cItem, 'colorHover', 'hover');
                    $class .= $this->colorClassFromColor($cItem, 'colorTxt', 'txt');
                    $class .= $this->colorClassFromColor($cItem, 'colorTxtHover', 'txt-hover');
                    $class .= $this->colorClassFromColor($cItem, 'colorBg', 'bg');
                    $class .= $this->colorClassFromColor($cItem, 'colorBgHover', 'bg-hover');

                    if(isset($cItem['target'])){
                        $class .= ' cusmin-target-' .$cItem['target'];
                    }

                    $icon = '';
                    //FA icon
                    //TODO: Cusmin adds multiple fa classes to the menu item
                    if (strpos($cItem['icon'], 'fa-') !== false) {
                        $class .= ' cusmin-fa ' . $cItem['icon'];
                    } else if (strpos($cItem['icon'], 'mi-') !== false) {
                        $class .= ' cusmin-mi ' . $cItem['icon'];
                    } //Probably WP icon
                    else {
                        $class .= ' cusmin-dashicon';
                        $icon = $cItem['icon'];
                    }

                    $updatedMenu[] = array(
                        $cItem['title'],
                        (isset($cItem['cap'])?$cItem['cap']:'read'),
                        $cItem['url'],
                        '',
                        $class,
                        '',
                        $icon
                    );
                    continue;
                }

                //find the same match in original admin menu
                foreach ((array) $menu as $oIndex => $oItem) {
                    if ($oItem[2] == $url) {
                        if (isset($cItem['icon']) && $cItem['icon']) {
                            //FA icon
                            if (strpos($cItem['icon'], 'fa-') !== false) {
                                $oItem[6] = '';
                                $oItem[4] .= ' cusmin-fa ' . $cItem['icon'];
                            }
                            //MI icon
                            else if (strpos($cItem['icon'], 'mi-') !== false) {
                                    $oItem[6] = '';
                                    $oItem[4] .= ' cusmin-mi ' . $cItem['icon'];
                            }
                            //Probably WP icon
                            else {
                                $oItem[6] = $cItem['icon'];
                                $oItem[4] .= ' cusmin-dashicon';
                            }

                            //Fix for 3rd party icons, mess their id
                            if(isset($oItem[5])){
                                $oItem[5].= "-none";
                            }


                            //Fix for third party icons, don't add their css
                            //$id = $oItem[5];
                            //$oItem[4] .=' cusmin-fa fa-vimeo';
                            //$oItem[4] = str_replace($id, '', $oItem[4]);
                            //$oItem[5] = '';
                        }
                        if (isset($cItem['title']) && $cItem['title']) {
                            $oItem[0] = $cItem['title'];
                        }
                        if (isset($cItem['cap']) && $cItem['cap']) {
                            //$oItem[1] = $cItem['cap'];
                        }
                        //TODO: Duplicate code from above, move to menu manager class and refactor
                        $oItem[4] .= $this->colorClassFromColor($cItem, 'color');
                        $oItem[4] .= $this->colorClassFromColor($cItem, 'colorHover', 'hover');
                        $oItem[4] .= $this->colorClassFromColor($cItem, 'colorTxt', 'txt');
                        $oItem[4] .= $this->colorClassFromColor($cItem, 'colorTxtHover', 'txt-hover');
                        $oItem[4] .= $this->colorClassFromColor($cItem, 'colorBg', 'bg');
                        $oItem[4] .= $this->colorClassFromColor($cItem, 'colorBgHover', 'bg-hover');
                        $updatedMenu[] = $oItem;
                        continue;
                    }
                }
            }
            //add the rest of the items from original menu (that are added after customizations)
            foreach ((array) $menu as $oIndex => $oItem) {
                if (preg_match('/separator/', $oItem[2])) {
                    continue;
                }
                $url = $oItem[2];
                if (!in_array($url, $coveredURLs)) {
                    $updatedMenu[] = $oItem;
                }
            }

            $menu = $updatedMenu;
        }
        //print_r($menu);die;
    }

    function adminMenuApplySub()
    {
        $updatedmenu_sub = array();
        global $submenu;

        $amCustomizations = $this->getAdminMenuUserCustomizations();
        if (isset($amCustomizations['sub'])) {
            $coveredTopURLs = array();

            //add sub-items from customizations
            foreach ($amCustomizations['sub'] as $topIndex => $subs) {
                $coveredTopURLs[] = $topIndex;
                if (!isset($updatedmenu_sub[$topIndex])) {
                    $updatedmenu_sub[$topIndex] = array();
                }
                if ($subs) {
                    $coveredSubIndexes = array();
                    $toRemoveSubIndexes = array();
                    foreach ($subs as $subIndex => $sub) {
                        $coveredSubIndexes[] = $subIndex;

                        $item = array();

                        //find original item
                        if (isset($submenu[$topIndex])) {
                            $foundSi = false;
                            foreach ($submenu[$topIndex] as $si => $s) {
                                if ($s[2] == $sub['url']) {
                                    $foundSi = true;
                                    //remove original item
                                    unset($submenu[$topIndex][$si]);
                                    //save a temporary item to be added later
                                    $item = $s;
                                    break;
                                } else if (strpos($sub['url'], 'customize.php') === 0) {
                                    $foundSi = true;
                                    //special case: customize.php url has additional query parameters
                                    //there are multiple of different sub items having this url, but with different
                                    //query parameters.
                                    $headerImage = '=header_image';
                                    $bkgImage = '=background_image';
                                    $oUrl = $s[2];
                                    $cUrl = $sub['url'];
                                    if (strpos($oUrl, 'customize.php') === 0) {
                                        //Header button
                                        if ((strpos($oUrl, $headerImage) !== false && strpos($cUrl, $headerImage) !== false)
                                            //Background button
                                            || (strpos($oUrl, $bkgImage) !== false && strpos($cUrl, $bkgImage) !== false)
                                            //Customize button
                                            || (
                                                strpos($oUrl, $headerImage) == false
                                                && strpos($oUrl, $bkgImage) == false
                                                && strpos($cUrl, $headerImage) == false
                                                && strpos($cUrl, $bkgImage) == false
                                            )
                                        ) {
                                            $sub['url'] = $s[2];
                                            unset($submenu[$topIndex][$si]);
                                            $item = $s;
                                            break;
                                        }
                                    }
                                }
                            }
                            if (!$foundSi) {
                                //remove non-custom items, 3rd party plugins removed from outside,
                                //but left their customizations
                                if (!isset($sub['custom'])) {
                                    $toRemoveSubIndexes[] = $subIndex;
                                }
                            }
                        }

                        if (isset($sub['hidden']) && $sub['hidden']) {
                            //skip
                            continue;
                        }

                        if (isset($sub['title'])) {
                            $item[0] = $sub['title'];
                            //TODO: Add feature for optional page name/title
                            $item[3] = $sub['title'];
                        }

                        $item[2] = $sub['url'];

                        if (isset($sub['cap'])) {
                            $item[1] = $sub['cap'];
                        }

                        if(!isset($item[0])){
                            $item[0] = '';
                        }

                        //remove from 3rd party plugins that are deactivated
                        if (!isset($sub['custom'])) {
                            if (in_array($subIndex, $toRemoveSubIndexes)) {
                                continue;
                            }
                        }

                        //add default capability in case that is not set
                        if (!isset($item[1]) || $item[1] == '') {
                            $item[1] = 'read';
                        }
                        if ($topIndex == 'index.php') {
                            //print_r($item);
                        }
                        /*  if (!isset($updatedmenu_sub[$topIndex][$subIndex])) {
                              $updatedmenu_sub[$topIndex][$subIndex] = array();
                          }*/
                        if(isset($sub['target'])){
                            if(!isset($item[4])){
                                $item[4] = '';
                            }
                            $item[4] .= ' cusmin-target-' .$sub['target'];
                        }
                        //print_r($submenu);die;
                        $updatedmenu_sub[$topIndex][$subIndex] = $item;
                    }

                    //go through original sub-items and add missing added after customizations
                    if (isset($submenu[$topIndex])) {
                        foreach ($submenu[$topIndex] as $si => $sub) {
                            if (!in_array($si, $coveredSubIndexes)) {
                                $updatedmenu_sub[$topIndex][$si] = $sub;
                            }
                        }
                    }
                }
            }
            //add additional subs for menu items added after customizations
            foreach ((array)$submenu as $topIndex => $subs) {
                if (!in_array($topIndex, $coveredTopURLs, true)) {
                    $updatedmenu_sub[$topIndex] = $subs;
                }
            }

            //print_r($submenu);
            //print_r($updatedmenu_sub);die;
            $submenu = $updatedmenu_sub;
        }
    }

    #region PRIVATE METHODS
    private static function validateConfigurationExists($configurationSlug)
    {
        $options = self::get();
        if (!isset($options[self::CONFIGURATIONS][$configurationSlug])) {
            if ($configurationSlug !== 'default') {
                throw new Exception('Configuration not found. Please choose different configuration.');
            }
        }
    }

    private static function createConfigurationIfDoesntExists($configurationSlug, $configurationName)
    {
        $options = self::get();
        if (!isset($options[self::CONFIGURATIONS][$configurationSlug])) {
            if ($configurationSlug !== 'default') {
                CusminOptions::addConfiguration($configurationSlug, $configurationName);
            }
        }
    }

    private static function newEmptyConfiguration($slug, $name)
    {
        $applyToRules = self::newApplyToRule();
        try {
            $applyToRules['ids'][] = get_current_user_id();
        }catch (\Exception $e){}

        return array(
            'slug' => $slug,
            'name' => $name,
            self::CUSTOMIZATION => array(),
            self::RULES => array(
                //self::ACCESS_BY => self::newAccessByRule(),
                self::APPLY_TO => $applyToRules
            )
        );
    }

    private static function newAccessByRule()
    {
        //Make sure that current user is always admin
        $newRule = self::newRule();
        $currentUserId = get_current_user_id();
        $newRule['ids'][] = $currentUserId;
        return $newRule;
    }

    private static function newApplyToRule()
    {
        return self::newRule();
    }

    private static function newRule()
    {
        //TODO: Add current user as default admin, none as
        //TODO: Create Cusmin Admins and Cusmin Users groups
        //that will be used in roles, for admins and users
        return array(
            /**
             * An array of user Ids
             */
            'ids' => array(),

            /**
             * An array of user groups
             */
            'groups' => array(),

            /**
             * An array of user capabilities
             */
            'capabilities' => array()
        );
    }
    #endregion
}